#!/bin/bash
#-----------------------------------------------------------------
# This script formats the LMDZ fortran routines with
# a common and unified format
# it works in 1 or 2 steps:
# 1/an optional preprocessing step with findent 
# (more robust for code structuration)
# 2/a formatting step with fprettify using a .fprettify format file
#
# if findent or fprettify displays an error or warning message,
# the file is not processed
# v1: November 2025
#-----------------------------------------------------------------


# List of files to process
#------------------------------------------------------------------
files_to_process=( $(find ../libf/*/* -maxdepth 0 -type f ! -xtype l \( -name "*.f90" -o -name "*.F90" \)) )

# Check installation of findent and fprettify 
#------------------------------------------------------------------
if command -v findent >/dev/null 2>&1; then
    echo "findent has been installed"
    findent --version
else
    echo "findent has not been installed"
    echo "please install it with sudo apt-get install findent"
fi


if command -v fprettify >/dev/null 2>&1; then
    echo "fprettify has been installed."
    fprettify --version
else
    echo "fprettify has not been installed."
    echo "please install it with: pip install fprettify"
fi

# write the .fprettify configuration file
#------------------------------------------------------------------
FILENAME=".fprettify"

# check if the .fprettify exists
if [ -f "$FILENAME" ]; then
    echo "the file $FILENAME already exists"
else
# create the file
cat << 'EOF' > "$FILENAME"
# --- Indentation ---
indent = 4                  # 4-space indentation (standard practice)
indent_cont = 6             # continuation lines more indented for clarity
tabindent = false           # always use spaces, never tabs

# --- Line formatting ---
max_line_length = 200       # good default for modern screens
whitespace = true           # trim trailing whitespace
whitespace_statements = true
whitespace_operators = true # enforce spaces around operators
compact = false             # do not compact everything (more readable)

# --- Keywords ---
case_keywords = lower       # lowercase keywords (if, do, end, module)
case_intrinsics = lower     # lowercase intrinsics (sin, cos, abs)
case_types = preserve       # preserve user-defined type casing

# --- Logical formatting ---
split_multiline_statements = true     # break long statements
align_assignments = true              # align "=" signs vertically
align_continuation = true

# --- Operators ---
space_in_percents = false   # Fortran % operator (component), no spaces
space_in_power = false      # no spaces around exponent (**)
space_in_colons = false     # array slices remain compact (i:j)
space_in_parentheses = false

# --- Comments ---
keep_comments = true
align_comments = true       # align trailing comments
comment_indent = 2

# --- Preprocessing ---
cpp_keywords = true         # format #ifdef / #define blocks
preserve_cpp_indentation = true

# --- Safety ---
strict_indent = true         # fail on inconsistent indentation
ignore_old_style = false     # do not preserve ugly fixed-format legacy stuff

EOF
fi

# formatting step
#------------------------------------------------------------------
# Array to store files that could not be processed
failures=()

# Function that applies findent and fprettify to a file
# with checks on errors and warnings

process_file() {
    local f="$1"
    local tmp
    tmp=$(mktemp) || return 1

    local patterns="warning|error|failed|deprecated"
    local err
    local ret

    cp "$f" "$tmp"

#    # Step 1: findent
#    rm "$tmp"
#    err=$(findent < "$f" 2>&1)
#    ret=$?
#    if [ $ret -ne 0 ] || echo "$err" | grep -Eqi "$patterns"; then
#        echo "findent detected WARNING/ERROR in $f"
#        rm -f "$tmp"
#        return 1
#    fi
#    findent < "$f" > "$tmp" 2>/dev/null


    # Step 2: fprettify
    err=$(fprettify "$tmp" 2>&1)
    ret=$?
    if [ $ret -ne 0 ] || echo "$err" | grep -Eqi "$patterns"; then
        echo "fprettify detected WARNING/ERROR in $f"
        rm -f "$tmp"
        return 1
    fi

    # Step 3: replace original file
    mv "$tmp" "$f"
    return 0
}


# Loop over all the files that we want to process
for f in "${files_to_process[@]}"; do
    echo "Processing $f ..."

    if process_file "$f"; then
        echo "ok"
    else
	echo "failure"
        failures+=("$f")
    fi
done

# Summary report
echo
echo "========== SUMMARY =========="
if [ ${#failures[@]} -eq 0 ]; then
    echo "All files were processed successfully."
else
    echo "The following files could NOT be processed:"
    for f in "${failures[@]}"; do
        echo " - $f"
    done
fi


