#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Novembre 2025

@author: Etienne Vignon

Modifications:

COMBLE original case definition
Forcing and initial conditions for 13 March 2020 COMBLE intercomparison case
"https://arm-development.github.io/comble-mip/
Timothy W. Juliano (NCAR/RAL, tjuliano@ucar.edu); Florian Tornow (NASA/GISS, ft2544@columbia.edu); Ann M. Fridlind (NASA/GISS, ann.fridlind@nasa.gov)
adapted to latest DEPHY format by E Vignon, L Raillard, R. Roehrig

"""

import os
import netCDF4 as nc
import numpy as np

from dephycf.Case import Case

################################################
# 0. General configuration of the present script
################################################

lplot = True # plot all the variables
lverbose = True # print information about variables and case

################################################
# 1. General information about the case
################################################

case = Case('COMBLE/REF',
        lat=74.5,
        lon=3.78,
        startDate="20200312220000",
        endDate="20200313180000",
        surfaceType="ocean",
        zorog=0.)

case.set_title("Forcing and initial conditions for COMBLE case - Original definition")
case.set_reference("https://arm-development.github.io/comble-mip/")
case.set_author("Timothy W. Juliano (NCAR/RAL, tjuliano@ucar.edu); Florian Tornow (NASA/GISS, ft2544@columbia.edu); Ann M. Fridlind (NASA/GISS, ann.fridlind@nasa.gov)")
case.set_script("driver_DEF.py")
case.set_comment("Use of original forcing file in previous DEPHY format COMBLE_INTERCOMPARISON_FORCING_V2.4.nc ")

################################################
# 2. Input netCDF file
################################################

fin = nc.Dataset('COMBLE_INTERCOMPARISON_FORCING_V2.4.nc','r')

################################################
# 3. Initial state
################################################

# vertical levels=
z=np.squeeze(fin['lev'][:])

# Surface pressure
ps  = np.squeeze(fin['ps'][0,0,0])
case.add_init_ps(ps)

# initial height
case.add_init_height(z,lev=z,levtype='altitude')

# Pressure
pressure  = np.squeeze(fin['pressure'][0,:,0,0])
case.add_init_pressure(pressure,lev=z,levtype='altitude')


# Temperature in K

temp  = np.squeeze(fin['temp'][0,:,0,0])
case.add_init_temp(temp,lev=z,levtype='altitude')
# Temperature in K and moisture (vapor and total mixing ratio)

theta  = np.squeeze(fin['theta'][0,:,0,0])
case.add_init_theta(theta,lev=z,levtype='altitude')


#  water vapor mixing ratio in kg/kg
qv =  np.squeeze(fin['qv'][0,:,0,0])
case.add_init_qv(qv,lev=z,levtype='altitude')

#  total water mixing ratio in kg/kg
qt=qv
case.add_init_qt(qt,lev=z,levtype='altitude')

# Zonal and meridional wind
u  = np.squeeze(fin['u'][0,:,0,0])
v  = np.squeeze(fin['v'][0,:,0,0])
case.add_init_wind(u=u,v=v,lev=z,levtype='altitude')


################################################
# 3. Forcing
################################################
time_forcing=fin['time'][:]

# longitude
lon=fin['lon_ref'][:]
case.add_longitude(lon,time=time_forcing,timeid='time')


# latitude
lat=fin['lat_ref'][:]
case.add_latitude(lat,time=time_forcing,timeid='time')


# SST [K]
ts  = np.squeeze(fin['ts'][:])
z0=0.0009
z0h=0.0000055
case.add_forcing_ts(ts,time=time_forcing,z0=z0, z0h=z0h)
case.add_forcing_thetas(ts,time=time_forcing,z0=z0, z0h=z0h)


# pressure 
surface_pressure=np.repeat(ps,len(time_forcing))
case.add_surface_pressure_forcing(surface_pressure, time=time_forcing)
pressure_forcing=np.tile(pressure,(len(time_forcing),1))
case.add_pressure_forcing(pressure_forcing,lev=z,levtype='altitude',time=time_forcing)

# ozone(convert kg/kg to mole fraction in air)

o3  = fin['o3'][0,:,0,0]
Md=28.96 #g/mol
Mo3=48.0 #g/mol
ro3=o3*Md/Mo3
ozone_forcing=np.tile(ro3,(len(time_forcing),1))
case.add_ozone(ro3=ozone_forcing,lev=z,levtype='altitude',time=time_forcing)


# geostrophic wind
ug  = fin['ug'][:,:]
vg  = fin['vg'][:,:]
case.add_geostrophic_wind(ug=ug,vg=vg,lev=z,time=time_forcing,levtype='altitude')

################################################
# 4. Writing file
################################################

case.write('COMBLE_REF_DEF_driver.nc')

if lverbose:
    case.info()

################################################
# 5. Ploting, if asked
################################################
if lplot:
    case.plot(rep_images='./images/driver_DEF/',timeunits='hours')

