#!/bin/bash

#-----------------------------------------------------------
# program that converts lmdz standard outputs to 
# DEPHY-compliant outputs
# Etienne Vignon, adaptation of a script by MP Lefebvre
# list of variable name convention is available here:
# https://docs.google.com/spreadsheets/d/1L4G2E7kDLkD-oMY1oiHSylJAGF22emW0-s1Q6PU5pX8/edit#gid=0
#-----------------------------------------------------------


# 1st step: add and calculate the required variables in the new output file

file_orig=hourly.nc


#ncdump -h $file_orig | grep -i float | sed -e 's/(/ /' | cut -d' ' -f2 >| tmp_list_var

list_vars="time_counter presnivs zfull zhalf pres paprs temp theta ovap oliq iwcon lwcon rhl rhi rhum vitu vitv LWdnSFC LWupSFC SWdnSFC SWupSFC \
flat evap evu ustar precip snow psol tsol t2m q2m  cldt rneb rneblsvol rsd rsu rld rlu f_th mc dmc \
dtthe dtswr dtlwr dtcon dtvdf tntr tnt dqthe dqcon dqvdf tnhus tntc tnhusc tke lwp iwp tke_dissip \
wape a_th w_th e_th d_th alp_wk q_th wake_deltat wake_deltaq wake_s wake_dens wake_omg wake_h"

\rm -f tmp.nc

ncks -v,temp $file_orig tmp.nc
for var in $list_vars ; do
  echo $var
  ncks -A -v $var $file_orig tmp.nc 
done

# calculate some variables

ncap2 -A -s 'zf=geop/9.81' -s 'rho=pres/(287.*temp*(1+0.61*ovap/(1.-ovap)))' -s 'WND=(vitu^2+vitv^2)^0.5'  -v $file_orig  tmp.nc
ncap2 -A -s 'rv=ovap/(1.-ovap)' -s 'qt=ovap+ocond' -s 'rt=qt/(1.-qt)'  -v $file_orig  tmp.nc
ncap2 -A -s 'hur=rhum*100.' -s 'hurl=rhl*100.' -s 'huri=rhi*100.' -v $file_orig  tmp.nc
ncap2 -A -s 'oice=ocond-oliq' -v  $file_orig  tmp.nc
ncap2 -A -s 'hs=sens*(-1.)' -v $file_orig  tmp.nc
ncap2 -A -s 'cwp=lwp+iwp' -v $file_orig  tmp.nc


if [ "`grep '^'dqlsc'$' tmp_list_var`" != "" ] ; then ncap2 -A -s 'dqtdt_ls=dqlsc+dqeva' -v $file_orig tmp.nc ; fi
if [ "`grep '^'dtlsc'$' tmp_list_var`" != "" ] ; then ncap2  -A -s 'dttdt_ls=dtlsc+dteva' -v $file_orig tmp.nc ; fi

ncatted -O -a _FillValue,,d,, tmp.nc


# 2nd step: rename  variables and dimensions with DEPHY compliant name (CF convention)
#           add the attributes


# Dimensions

ncrename -d time_counter,time -v time_counter,time tmp.nc
ncatted -a standard_name,time,o,c,'time' tmp.nc
ncatted  -a calendar,time,o,c,'365_day' tmp.nc
#ncatted -a units,time,o,c,'seconds since YYY-MM-DD HH:MM:SS' tmp.nc


ncrename -d presnivs,levf -v presnivs,levf tmp.nc
ncatted -a standard_name,levf,o,c,'Full-level' tmp.nc
ncatted -a long_name,levf,o,c,'Full-level' tmp.nc

# standard variables 

ncrename -d zf,zf tmp.nc
ncatted -a standard_name,zf,o,c,'Full-level height' tmp.nc
ncatted -a long_name,zf,o,c,'Full-level height' tmp.nc
ncatted -a units,zf,o,c,'m' tmp.nc

ncrename -d zhalf,zh tmp.nc
ncatted -a standard_name,zh,o,c,'Half-level height' tmp.nc
ncatted -a long_name,zh,o,c,'Half-level height' tmp.nc
ncatted -a units,zh,o,c,'m' tmp.nc

ncrename -v pres,pa tmp.nc
ncatted -a standard_name,pf,o,c,'air_pressure' tmp.nc
ncatted -a long_name,pf,o,c,'air_pressure' tmp.nc
ncatted -a units,pf,o,c,'Pa' tmp.nc

ncrename -v paprs,ph tmp.nc
ncatted -a standard_name,ph,o,c,'Half-level air_pressure' tmp.nc
ncatted -a long_name,ph,o,c,'Half-level air_pressure' tmp.nc
ncatted -a units,ph,o,c,'Pa' tmp.nc

ncrename -v rho,rho tmp.nc
ncatted -a standard_name,rho,o,c,'air_volumic_mass' tmp.nc
ncatted -a long_name,rho,o,c,'air_volumic_mass' tmp.nc
ncatted -a units,rho,o,c,'kg m-3' tmp.nc

ncrename -v temp,ta tmp.nc
ncatted -a standard_name,ta,o,c,'air_temperature' tmp.nc
ncatted -a long_name,ta,o,c,'air_temperature' tmp.nc
ncatted -a units,ta,o,c,'K' tmp.nc

ncrename -v ovap,qv tmp.nc
ncatted -a standard_name,qv,o,c,'specific_humidity' tmp.nc
ncatted -a long_name,qv,o,c,'specific_humidity' tmp.nc
ncatted -a units,qv,o,c,'1' tmp.nc

ncrename -v oliq,ql tmp.nc
ncatted -a standard_name,ql,o,c,'mass_fraction_of_cloud_liquid_water_in_air' tmp.nc
ncatted -a long_name,ql,o,c,'mass_fraction_of_cloud_liquid_water_in_air' tmp.nc
ncatted -a units,ql,o,c,'1' tmp.nc

ncrename -v oice,qi tmp.nc
ncatted -a standard_name,qi,o,c,'mass_fraction_of_cloud_ice_water_in_air' tmp.nc
ncatted -a long_name,qi,o,c,'mass_fraction_of_cloud_ice_water_in_air' tmp.nc
ncatted -a units,qi,o,c,'1' tmp.nc

ncrename -v lwcon,qlrad tmp.nc
ncatted -a standard_name,qlrad,o,c,'Radiatively-active liquid water content' tmp.nc
ncatted -a long_name,qlrad,o,c,'Radiatively-active liquid water content' tmp.nc
ncatted -a units,qlrad,o,c,'1' tmp.nc

ncrename -v iwcon,qirad tmp.nc
ncatted -a standard_name,qirad,o,c,'Radiatively-active ice water content' tmp.nc
ncatted -a long_name,qirad,o,c,'Radiatively-active ice water content' tmp.nc
ncatted -a units,qirad,o,c,'1' tmp.nc

ncrename -v rv,rv tmp.nc
ncatted -a standard_name,rv,o,c,'humidity_mixing_ratio' tmp.nc
ncatted -a long_name,rv,o,c,'humidity_mixing_ratio' tmp.nc
ncatted -a units,rv,o,c,'1' tmp.nc

ncrename -v qt,qt tmp.nc
ncatted -a standard_name,qt,o,c,'Total water content' tmp.nc
ncatted -a long_name,qt,o,c,'Total water content' tmp.nc
ncatted -a units,qt,o,c,'kg kg-1' tmp.nc

ncrename -v rt,rt tmp.nc
ncatted -a standard_name,rt,o,c,'Total water mixing ratio' tmp.nc
ncatted -a long_name,rt,o,c,'Total water mixing ratio' tmp.nc
ncatted -a units,rt,o,c,'kg kg-1' tmp.nc

ncrename -v hur,hur tmp.nc
ncatted -a standard_name,hur,o,c,'relative_humidity' tmp.nc
ncatted -a long_name,hur,o,c,'relative_humidity' tmp.nc
ncatted -a units,hur,o,c,'%' tmp.nc

ncrename -v hurl,hurl tmp.nc
ncatted -a standard_name,hurl,o,c,'relative_humidity_wrt_liquid' tmp.nc
ncatted -a long_name,hurl,o,c,'relative_humidity_wrt_liquid' tmp.nc
ncatted -a units,hurl,o,c,'%' tmp.nc

ncrename -v huri,huri tmp.nc
ncatted -a standard_name,huri,o,c,'relative_humidity_wrt_ice' tmp.nc
ncatted -a long_name,huri,o,c,'relative_humidity_wrt_ice' tmp.nc
ncatted -a units,huri,o,c,'%' tmp.nc

ncrename -v vitu,ua tmp.nc
ncatted -a standard_name,ua,o,c,'eastward_wind' tmp.nc
ncatted -a long_name,ua,o,c,'eastward_wind' tmp.nc
ncatted -a units,ua,o,c,'m s-1' tmp.nc

ncrename -v vitv,va tmp.nc
ncatted -a standard_name,va,o,c,'northward_wind' tmp.nc
ncatted -a long_name,va,o,c,'northward_wind' tmp.nc
ncatted -a units,va,o,c,'m s-1' tmp.nc

ncrename -v vitv,va tmp.nc
ncatted -a standard_name,va,o,c,'northward_wind' tmp.nc
ncatted -a long_name,va,o,c,'northward_wind' tmp.nc
ncatted -a units,va,o,c,'m s-1' tmp.nc

ncrename -v tke,tke tmp.nc
ncatted -a standard_name,tke,o,c,'specific_turbulent_kinetic_energy' tmp.nc
ncatted -a long_name,tke,o,c,'specific_turbulent_kinetic_energy' tmp.nc
ncatted -a units,tke,o,c,'m2 s-2' tmp.nc

ncrename -v tke_dissip,tkediss tmp.nc
ncatted -a standard_name,tkediss,o,c,'specific_turbulent_kinetic_energy_dissipation' tmp.nc
ncatted -a long_name,tkediss,o,c,'specific_turbulent_kinetic_energy_dissipation' tmp.nc
ncatted -a units,tkediss,o,c,'m2 s-3' tmp.nc

# surface fluxes

ncrename -v LWdnSFC,rlds tmp.nc
ncatted -a standard_name,rlds,o,c,'surface_downwelling_longwave_flux_in_air' tmp.nc
ncatted -a long_name,rlds,o,c,'surface_downwelling_longwave_flux_in_air' tmp.nc
ncatted -a units,rlds,o,c,'W m-2' tmp.nc

ncrename -v LWupSFC,rlus tmp.nc
ncatted -a standard_name,rlus,o,c,'surface_upwelling_longwave_flux_in_air' tmp.nc
ncatted -a long_name,rlus,o,c,'surface_upwelling_longwave_flux_in_air' tmp.nc
ncatted -a units,rlus,o,c,'W m-2' tmp.nc

ncrename -v SWdnSFC,rsds tmp.nc
ncatted -a standard_name,rsds,o,c,'surface_downwelling_shortwave_flux_in_air' tmp.nc
ncatted -a long_name,rsds,o,c,'surface_downwelling_shortwave_flux_in_air' tmp.nc
ncatted -a units,rsds,o,c,'W m-2' tmp.nc

ncrename -v SWupSFC,rsus tmp.nc
ncatted -a standard_name,rsus,o,c,'surface_upwelling_shortwave_flux_in_air' tmp.nc
ncatted -a long_name,rsus,o,c,'surface_upwelling_shortwave_flux_in_air' tmp.nc
ncatted -a units,rsus,o,c,'W m-2' tmp.nc

ncrename -v hs,hfss tmp.nc
ncatted -a standard_name,hfss,o,c,'surface_upward_sensible_heat_flux' tmp.nc
ncatted -a long_name,hfss,o,c,'surface_upward_sensible_heat_flux' tmp.nc
ncatted -a units,hfss,o,c,'W m-2' tmp.nc

ncrename -v flat,hfls tmp.nc
ncatted -a standard_name,hfls,o,c,'surface_upward_latent_heat_flux' tmp.nc
ncatted -a long_name,hfls,o,c,'surface_upward_latent_heat_flux' tmp.nc
ncatted -a units,hfls,o,c,'W m-2' tmp.nc

ncrename -v ustar,ustar tmp.nc
ncatted -a standard_name,ustar,o,c,'surface_friction_velocity' tmp.nc
ncatted -a long_name,ustar,o,c,'surface_friction_velocity' tmp.nc
ncatted -a units,ustar,o,c,'m s-1' tmp.nc

# radiative fluxes

ncrename -v rld,rld tmp.nc
ncatted -a standard_name,rld,o,c,'downwelling_longwave_flux_in_air' tmp.nc
ncatted -a long_name,rld,o,c,'downwelling_longwave_flux_in_air' tmp.nc
ncatted -a units,rld,o,c,'W m-2' tmp.nc

ncrename -v rlu,rlu tmp.nc
ncatted -a standard_name,rlu,o,c,'upwelling_longwave_flux_in_air' tmp.nc
ncatted -a long_name,rlu,o,c,'upwelling_longwave_flux_in_air' tmp.nc
ncatted -a units,rlu,o,c,'W m-2' tmp.nc

ncrename -v rsd,rsd tmp.nc
ncatted -a standard_name,rsd,o,c,'downwelling_shortwave_flux_in_air' tmp.nc
ncatted -a long_name,rsd,o,c,'downwelling_shortwave_flux_in_air' tmp.nc
ncatted -a units,rsd,o,c,'W m-2' tmp.nc

ncrename -v rsu,rsu tmp.nc
ncatted -a standard_name,rsu,o,c,'upwelling_shortwave_flux_in_air' tmp.nc
ncatted -a long_name,rsu,o,c,'upwelling_shortwave_flux_in_air' tmp.nc
ncatted -a units,rsu,o,c,'W m-2' tmp.nc

# precip and cloud

ncrename -v precip,pr tmp.nc
ncatted -a standard_name,pr,o,c,'precipitation_flux' tmp.nc
ncatted -a long_name,pr,o,c,'precipitation_flux' tmp.nc
ncatted -a units,pr,o,c,'kg m-2 s-1' tmp.nc

ncrename -v snow,prsn tmp.nc
ncatted -a standard_name,prsn,o,c,'snowfall_flux' tmp.nc
ncatted -a long_name,prsn,o,c,'snowfall_flux' tmp.nc
ncatted -a units,prsn,o,c,'kg m-2 s-1' tmp.nc

ncrename -v lwp,lwp tmp.nc
ncatted -a standard_name,lwp,o,c,'atmospheric_liquid_water_content' tmp.nc
ncatted -a long_name,lwp,o,c,'atmospheric_liquid_water_content' tmp.nc
ncatted -a units,lwp,o,c,'1' tmp.nc

ncrename -v iwp,iwp tmp.nc
ncatted -a standard_name,iwp,o,c,'atmospheric_ice_water_content' tmp.nc
ncatted -a long_name,iwp,o,c,'atmospheric_ice_water_content' tmp.nc
ncatted -a units,iwp,o,c,'1' tmp.nc

ncrename -v cwp,cwp tmp.nc
ncatted -a standard_name,cwp,o,c,'atmospheric_condensed_water_content' tmp.nc
ncatted -a long_name,cwp,o,c,'atmospheric_condensed_water_content' tmp.nc
ncatted -a units,cwp,o,c,'1' tmp.nc

ncrename -v rneb,cl tmp.nc
ncatted -a standard_name,cl,o,c,'cloud_area_fraction_in_atmosphere_layer' tmp.nc
ncatted -a long_name,cwp,o,c,'atmospheric_condensed_water_content' tmp.nc
ncatted -a units,cl,o,c,'1' tmp.nc

ncrename -v cldt,clt tmp.nc
ncatted -a standard_name,clt,o,c,'cloud_area_fraction' tmp.nc
ncatted -a long_name,clt,o,c,'cloud_area_fraction' tmp.nc
ncatted -a units,clt,o,c,'1' tmp.nc


# surface variable

ncrename -v psol,ps tmp.nc
ncatted -a standard_name,ps,o,c,'surface_pressure' tmp.nc
ncatted -a long_name,ps,o,c,'surface_pressure' tmp.nc
ncatted -a units,ps,o,c,'Pa' tmp.nc

ncrename -v tsol,ts tmp.nc
ncatted -a standard_name,ts,o,c,'surface_temperature' tmp.nc
ncatted -a long_name,ts,o,c,'surface_temperature' tmp.nc
ncatted -a units,ts,o,c,'K' tmp.nc

ncrename -v t2m,tas tmp.nc
ncatted -a standard_name,tas,o,c,'2m_air_temperature' tmp.nc
ncatted -a long_name,tas,o,c,'2m_air_temperature' tmp.nc
ncatted -a units,tas,o,c,'K' tmp.nc

ncrename -v q2m,qvs tmp.nc
ncatted -a standard_name,qvs,o,c,'2m_specific_humidity' tmp.nc
ncatted -a long_name,tas,o,c,'2m_air_temperature' tmp.nc
ncatted -a units,qvs,o,c,'1' tmp.nc


# convection and wakes  

ncrename -v a_th,alpha_up_sh tmp.nc
ncatted -a standard_name,alpha_up_sh,o,c,'shallow_convective_updraft_fraction' tmp.nc
ncatted -a long_name,alpha_up_sh,o,c,'shallow_convective_updraft_fraction' tmp.nc
ncatted -a units,alpha_up_sh,o,c,'1' tmp.nc

ncrename -v f_th,mf_up_shconv tmp.nc
ncatted -a standard_name,mf_up_shconv,o,c,'shallow_convective_updraft_mass_flux' tmp.nc
ncatted -a long_name,mf_up_shconv,o,c,'shallow_convective_updraft_mass_flux' tmp.nc
ncatted -a units,Mf_up_sh,o,c,'kg m-2 s-1' tmp.nc

ncrename -v w_th,wa_up_shconv tmp.nc
ncatted -a standard_name,wa_up_shconv,o,c,'shallow_convective_updraft_vertical_velocity' tmp.nc
ncatted -a long_name,wa_up_shconv,o,c,'shallow_convective_updraft_vertical_velocity' tmp.nc
ncatted -a units,wa_up_shconv,o,c,'m s-1' tmp.nc

ncrename -v e_th,ent_up_shconv tmp.nc
ncatted -a standard_name,ent_up_shconv,o,c,'shallow_convective_entrainment' tmp.nc
ncatted -a long_name,ent_up_shconv,o,c,'shallow_convective_entrainment' tmp.nc
ncatted -a units,ent_up_shconv,o,c,'m-1' tmp.nc

ncrename -v d_th,det_up_shconv tmp.nc
ncatted -a standard_name,det_up_shconv,o,c,'shallow_convective_detrainment' tmp.nc
ncatted -a long_name,det_up_shconv,o,c,'shallow_convective_detrainment' tmp.nc
ncatted -a units,det_up_shconv,o,c,'m-1' tmp.nc

ncrename -v q_th,qv_up_shconv tmp.nc
ncatted -a standard_name,qv_up_shconv,o,c,'shallow_convective_updraft_specific_humidity' tmp.nc
ncatted -a long_name,qv_up_shconv,o,c,'shallow_convective_updraft_specific_humidity' tmp.nc
ncatted -a units,qv_up_shconv,o,c,'1' tmp.nc


ncrename -v mc,mf_up tmp.nc
ncatted -a standard_name,mf_up,o,c,'convective_updraft_mass_flux' tmp.nc
ncatted -a long_name,mf_up,o,c,'convective_updraft_mass_flux' tmp.nc
ncatted -a units,mf_up,o,c,'kg m-2 s-1' tmp.nc

ncrename -v dmc,mf_up_dconv tmp.nc
ncatted -a standard_name,mf_up_dconv,o,c,'deep_convection_updraft_mass_flux' tmp.nc
ncatted -a long_name,mf_up_dconv,o,c,'deep_convection_updraft_mass_flux' tmp.nc
ncatted -a units,mf_up_dconv,o,c,'kg m-2 s-1' tmp.nc


ncrename -v wape,wape_pe tmp.nc
ncatted -a standard_name,wape_pe,o,c,'Wake potential energy' tmp.nc
ncatted -a long_name,wape_pe,o,c,'Wake potential energy' tmp.nc
ncatted -a units,wape_pe,o,c,'J kg-1' tmp.nc

ncrename -v alp_wk,wake_alp tmp.nc
ncatted -a standard_name,wake_alp,o,c,'wake_available_lifting_power' tmp.nc
ncatted -a long_name,wape_alp,o,c,'wake_available_lifting_power' tmp.nc
ncatted -a units,wape_alp,o,c,'W m-2' tmp.nc

ncrename -v wake_deltat,delta_ta tmp.nc
ncatted -a standard_name,delta_ta,o,c,'air_temperature_difference_between_wake_and_offwake_regions' tmp.nc
ncatted -a long_name,delta_ta,o,c,'air_temperature_difference_between_wake_and_offwake_regions' tmp.nc
ncatted -a units,delta_ta,o,c,'K' tmp.nc

ncrename -v wake_deltaq,delta_qv tmp.nc
ncatted -a standard_name,delta_qv,o,c,'specific_humidity_difference_between_wake_and_offwake_regions' tmp.nc
ncatted -a long_name,delta_qv,o,c,'specific_humidity_difference_between_wake_and_offwake_regions' tmp.nc
ncatted -a units,delta_qv,o,c,'1' tmp.nc

ncrename -v wake_s,sigmaw tmp.nc
ncatted -a standard_name,sigmaw,o,c,'wake_fractional_area' tmp.nc
ncatted -a long_name,sigmaw,o,c,'wake_fractional_area' tmp.nc
ncatted -a units,sigmaw,o,c,'1' tmp.nc

ncrename -v wake_dens,wdens tmp.nc
ncatted -a standard_name,wdens,o,c,'wake_density' tmp.nc
ncatted -a long_name,wdens,o,c,'wake_density' tmp.nc
ncatted -a units,wdens,o,c,'1' tmp.nc

ncrename -v wake_h,hw tmp.nc
ncatted -a standard_name,hw,o,c,'wake_depth' tmp.nc
ncatted -a long_name,hw,o,c,'wake_depth' tmp.nc
ncatted -a units,hw,o,c,'m' tmp.nc

ncrename -v wake_omg,wake_wap tmp.nc
ncatted -a standard_name,wake_wap,o,c,'wake_vertical_velocity' tmp.nc
ncatted -a long_name,wake_wap,o,c,'wake_vertical_velocity' tmp.nc
ncatted -a units,wake_wap,o,c,'Pa s-1' tmp.nc

# tendencies

ncrename -v dtswr,tntarsw tmp.nc
ncatted -a standard_name,tntarsw,o,c,'air_temperature_tendency_sw_radiation' tmp.nc
ncatted -a long_name,tntarsw,o,c,'air_temperature_tendency_sw_radiation' tmp.nc
ncatted -a units,tntarsw,o,c,'K s-1' tmp.nc

ncrename -v dtlwr,tntarlw tmp.nc
ncatted -a standard_name,tntarlw,o,c,'air_temperature_tendency_lw_radiation' tmp.nc
ncatted -a long_name,tntarlw,o,c,'air_temperature_tendency_lw_radiation' tmp.nc
ncatted -a units,tntarlw,o,c,'K s-1' tmp.nc

ncrename -v tntr,tntar tmp.nc
ncatted -a standard_name,tntar,o,c,'air_temperature_tendency_radiation' tmp.nc
ncatted -a long_name,tntar,o,c,'air_temperature_tendency_radiation' tmp.nc
ncatted -a units,tntar,o,c,'K s-1' tmp.nc

ncrename -v dtcon,tntadc tmp.nc
ncatted -a standard_name,tntadc,o,c,'air_temperature_tendency_deep_convection' tmp.nc
ncatted -a long_name,tntar,o,c,'air_temperature_tendency_radiation' tmp.nc
ncatted -a units,tntadc,o,c,'K s-1' tmp.nc

ncrename -v dtthe,tntashc tmp.nc
ncatted -a standard_name,tntashc,o,c,'air_temperature_tendency_shallow_convection' tmp.nc
ncatted -a long_name,tntashc,o,c,'air_temperature_tendency_shallow_convection' tmp.nc
ncatted -a units,tntashc,o,c,'K s-1' tmp.nc

ncrename -v tntc,tntac tmp.nc
ncatted -a standard_name,tntac,o,c,'air_temperature_tendency_convection' tmp.nc
ncatted -a long_name,tntac,o,c,'air_temperature_tendency_convection' tmp.nc
ncatted -a units,tntac,o,c,'K s-1' tmp.nc

ncrename -v dt_ls,tntalscp tmp.nc
ncatted -a standard_name,tntalscp,o,c,'air_temperature_tendency_large_scale_condensation' tmp.nc
ncatted -a long_name,tntalscp,o,c,'air_temperature_tendency_large_scale_condensation' tmp.nc
ncatted -a units,tntalscp,o,c,'K s-1' tmp.nc

ncrename -v dtvdf,tntapbl tmp.nc
ncatted -a standard_name,tntapbl,o,c,'air_temperature_tendency_turbulence' tmp.nc
ncatted -a long_name,tntapbl,o,c,'air_temperature_tendency_turbulence' tmp.nc
ncatted -a units,tntapbl,o,c,'K s-1' tmp.nc

ncrename -v tnt,tnta tmp.nc
ncatted -a standard_name,tnta,o,c,'air_temperature_tendency_total' tmp.nc
ncatted -a long_name,tnta,o,c,'air_temperature_tendency_total' tmp.nc
ncatted -a units,tnta,o,c,'K s-1' tmp.nc

ncrename -v dqcon,tnqvdc tmp.nc
ncatted -a standard_name,tnqvdc,o,c,'specific_humidity_tendency_deep_convection' tmp.nc
ncatted -a long_name,tnqvdc,o,c,'specific_humidity_tendency_deep_convection' tmp.nc
ncatted -a units,tnqvdc,o,c,'kg kg-1 s-1' tmp.nc

ncrename -v dqthe,tnqvshc tmp.nc
ncatted -a standard_name,tnqvshc,o,c,'specific_humidity_tendency_shallow_convection' tmp.nc
ncatted -a units,tnqvshc,o,c,'kg kg-1 s-1' tmp.nc

ncrename -v tnhusc,tnqvc tmp.nc
ncatted -a standard_name,tnqvc,o,c,'specific_humidity_tendency_convection' tmp.nc
ncatted -a long_name,tnqvc,o,c,'specific_humidity_tendency_convection' tmp.nc
ncatted -a units,tnqvc,o,c,'kg kg-1 s-1' tmp.nc

ncrename -v dqvdf,tnqvpbl tmp.nc
ncatted -a standard_name,tnqvpbl,o,c,'specific_humidity_tendency_turbulence' tmp.nc
ncatted -a long_name,tnqvpbl,o,c,'specific_humidity_tendency_turbulence' tmp.nc
ncatted -a units,tnqvpbl,o,c,'kg kg-1 s-1' tmp.nc

ncrename -v dq_ls,tnqvlscp tmp.nc
ncatted -a standard_name,tnqvlscp,o,c,'specific_humidity_tendency_large_scale_condensation' tmp.nc
ncatted -a long_name,tnqvlscp,o,c,'specific_humidity_tendency_large_scale_condensation' tmp.nc
ncatted -a units,tnqvlscp,o,c,'kg kg-1 s-1' tmp.nc

ncrename -v tnhus,tnqv tmp.nc
ncatted -a standard_name,tnqv,o,c,'specific_humidity_tendency_total' tmp.nc
ncatted -a long_name,tnqv,o,c,'specific_humidity_tendency_total' tmp.nc
ncatted -a units,tnqv,o,c,'kg kg-1 s-1' tmp.nc


# remove singleton dimensions lat and lon

ncwa -O -a lat tmp.nc tmp.nc
ncwa -O -a lon tmp.nc tmp.nc


# general attributes

ncatted -a author,global,o,c,'MP Lefebvre, E Vignon' tmp.nc
ncatted -h -a history,global,o,c,' ' tmp.nc
ncatted -h -a history_of_appended_files,global,d,, tmp.nc
ncatted -a physics,global,o,c,`pwd` tmp.nc


# rename file 

mv tmp.nc hourly_std_CF.nc

