MODULE qsat_seawater_mod

CONTAINS

!-------------------------------------------------------------------------------
!
!     ######################################
      FUNCTION QSAT_SEAWATER(knon, klon, PT,PP)
!$gpum horizontal knon

!     ######################################
!
!!****  *QSATW * - function to compute saturation vapor humidity from
!!                 temperature
!!
!!    PURPOSE
!!    -------
!       The purpose of this function is to compute the saturation vapor 
!     pressure from temperature over saline seawater
!      
!
!!**  METHOD
!!    ------
!!       Given temperature T (PT), the saturation vapor pressure es(T)
!!    (FOES(PT)) is computed by integration of the Clapeyron equation
!!    from the triple point temperature Tt (XTT) and the saturation vapor 
!!    pressure of the triple point es(Tt) (XESTT), i.e  
!!    The reduction due to salinity is compute with the factor 0.98 (reduction of 2%)
!!     
!!         es(T)= 0.98*EXP( alphaw - betaw /T - gammaw Log(T) )
!!  
!!     with :
!!       alphaw (XALPW) = LOG(es(Tt))+ betaw/Tt + gammaw Log(Tt) 
!!       betaw (XBETAW) = Lv(Tt)/Rv + gammaw Tt
!!       gammaw (XGAMW) = (Cl -Cpv) /Rv
!!
!!      Then, the specific humidity at saturation is deduced.
!!  
!!
!!    EXTERNAL
!!    --------
!!      NONE
!!
!!    IMPLICIT ARGUMENTS
!!    ------------------
!!      Module MODD_CST : comtains physical constants
!!        XALPW   : Constant for saturation vapor pressure function
!!        XBETAW  : Constant for saturation vapor pressure function
!!        XGAMW   : Constant for saturation vapor pressure function  
!!      
!!    REFERENCE
!!    ---------
!!      Book2 of documentation of Meso-NH 
!!      Zeng, X., Zhao, M., and Dickinson, R. E., 1998 : Intercomparaison of bulk
!!      aerodynamic algorithm for the computation of sea surface fluxes using
!!      TOGA COARE and TAO data. Journal of Climate, vol 11, nb 10, pp 2628--2644
!!
!!
!!    AUTHOR
!!    ------
!!      C. Lebeaupin    * Meteo France *
!!
!!    MODIFICATIONS
!!    -------------
!!      Original    6/04/2005 
!-------------------------------------------------------------------------------
!
!*       0.    DECLARATIONS
!              ------------
!
USE MODD_CSTS
USE indice_sol_mod
!
IMPLICIT NONE
!
!*       0.1   Declarations of arguments and results
!
!
INTEGER, INTENT(IN) :: knon  ! horizontal indice compressed
INTEGER, INTENT(IN) :: klon  ! horizontal indice (fake can be 1)

REAL, DIMENSION(klon), INTENT(IN)                :: PT     ! Temperature
                                                        ! (Kelvin)
REAL, DIMENSION(klon), INTENT(IN)                :: PP     ! Pressure
                                                        ! (Pa)
REAL, DIMENSION(SIZE(PT))                        :: PQSAT  ! saturation vapor 
                                                        ! specific humidity
                                                        ! with respect to
                                                        ! water (kg/kg)

!
!*       0.2   Declarations of local variables
!
REAL, DIMENSION(SIZE(PT))                   :: ZFOES  ! saturation vapor 
                                                        ! pressure
                                                        ! (Pascal) 
REAL :: QSAT_SEAWATER                                                        
!
REAL, DIMENSION(SIZE(PT))                   :: ZWORK1
REAL                                        :: ZWORK2
!REAL(KIND=JPRB) :: ZHOOK_HANDLE
!-------------------------------------------------------------------------------
!
!IF (LHOOK) CALL DR_HOOK('MODE_THERMOS:QSATSEAW_1D',0,ZHOOK_HANDLE)
!
!ZFOES  = 1 !PSAT(PT(:))
!ZFOES  = 0.98*ZFOES
!ZFOES(:) = ZFOES(:)*1013.25E+02             !convert from atm to Pa
ZFOES = 0.98*EXP( XALPW - XBETAW/PT - XGAMW*LOG(PT)  )
! vapor pressure reduction of 2% over saline seawater could have a significant 
! impact on the computation of surface latent heat flux under strong wind 
! conditions (Zeng et al, 1998). 
!
ZWORK1 = ZFOES/PP
ZWORK2    = XRD/XRV
!
!*       2.    COMPUTE SATURATION HUMIDITY
!              ---------------------------
!
PQSAT = ZWORK2*ZWORK1 / (1.+(ZWORK2-1.)*ZWORK1)
!
!IF (LHOOK) CALL DR_HOOK('MODE_THERMOS:QSATSEAW_1D',1,ZHOOK_HANDLE)
!-------------------------------------------------------------------------------
!
END FUNCTION QSAT_SEAWATER
!
!-------------------------------------------------------------------------------
END MODULE qsat_seawater_mod
